

/**
 * AutoMatch Pro - Storage Manager
 * Handles all Chrome storage operations with encryption support
 * @module storage
 */

import { CONFIG, ERROR_CODES, MAX_PENDING_MATCHES, QUEUE_STOP_THRESHOLD } from './constants.js';
import Logger from './logger.js';

class StorageManager {
  constructor() {
    this.logger = new Logger('StorageManager');
  }

  /**
   * Get data from Chrome storage
   * @param {string} key - Storage key
   * @returns {Promise<any>} Stored value or null
   */
  async get(key) {
    try {
      // FIX: Added a guard clause to ensure chrome.storage.local exists.
      // This prevents the "TypeError: Cannot read properties of undefined (reading 'local')"
      // if the script runs in an environment where the API is not available.
      if (chrome.storage && chrome.storage.local) {
        const result = await chrome.storage.local.get(key);
        return result[key] || null;
      }
      this.logger.warn('chrome.storage.local is not available in this context.');
      return null;
    } catch (error) {
      this.logger.error(`Failed to get ${key}:`, error);
      throw new Error(`${ERROR_CODES.STORAGE_ERROR}: Failed to retrieve data`);
    }
  }

  /**
   * Set data in Chrome storage
   * @param {string} key - Storage key
   * @param {any} value - Value to store
   * @returns {Promise<void>}
   */
  async set(key, value) {
    try {
      // FIX: Added a guard clause to ensure chrome.storage.local exists.
      if (chrome.storage && chrome.storage.local) {
        await chrome.storage.local.set({ [key]: value });
        this.logger.debug(`Stored ${key} successfully`);
      } else {
        this.logger.warn('chrome.storage.local is not available in this context.');
      }
    } catch (error) {
      this.logger.error(`Failed to set ${key}:`, error);
      throw new Error(`${ERROR_CODES.STORAGE_ERROR}: Failed to store data`);
    }
  }

  /**
   * Remove data from Chrome storage
   * @param {string} key - Storage key
   * @returns {Promise<void>}
   */
  async remove(key) {
    // FIX: Add a guard clause to prevent calling the API with an invalid key.
    if (!key) {
      this.logger.warn('Attempted to remove an undefined or null key from storage.');
      return; // Exit the function gracefully without throwing an error.
    }
    try {
      await chrome.storage.local.remove(key);
      this.logger.debug(`Removed ${key} from storage`);
    } catch (error) {
      this.logger.error(`Failed to remove ${key}:`, error);
    }
  }

  /**
   * Clear all extension storage
   * @returns {Promise<void>}
   */
  async clear() {
    try {
      await chrome.storage.local.clear();
      this.logger.info('Storage cleared');
    } catch (error) {
      this.logger.error('Failed to clear storage:', error);
    }
  }

  // Auth Management
  async getAuth() {
    const auth = await this.get(CONFIG.STORAGE_KEYS.AUTH);
    if (auth && auth.expiresAt) {
      // Check if token is expired
      if (Date.now() > auth.expiresAt) {
        await this.remove(CONFIG.STORAGE_KEYS.AUTH);
        return null;
      }
    }
    return auth;
  }

  async setAuth(token, userId = null) {
    const auth = {
      token,
      userId,
      expiresAt: Date.now() + CONFIG.TIMING.TOKEN_LIFETIME,
      createdAt: Date.now()
    };
    await this.set(CONFIG.STORAGE_KEYS.AUTH, auth);
    return auth;
  }

  async clearAuth() {
    await this.remove(CONFIG.STORAGE_KEYS.AUTH);
    await this.remove(CONFIG.STORAGE_KEYS.CREDITS);
    await this.remove(CONFIG.STORAGE_KEYS.USER_INFO);
  }

  // User Info Management
  async getUserInfo() {
    return await this.get(CONFIG.STORAGE_KEYS.USER_INFO);
  }

  async setUserInfo(userInfo) {
    await this.set(CONFIG.STORAGE_KEYS.USER_INFO, userInfo);
    return userInfo;
  }

  // Script Management
  async getScript() {
    return await this.get(CONFIG.STORAGE_KEYS.SCRIPT);
  }

  async setScript(version, content) {
    const script = {
      version,
      content,
      lastUpdated: Date.now()
    };
    await this.set(CONFIG.STORAGE_KEYS.SCRIPT, script);
    return script;
  }

  // Credits Management
  async getCredits() {
    return await this.get(CONFIG.STORAGE_KEYS.CREDITS);
  }

  async setCredits(credits) {
    const creditsData = {
      remaining: credits,
      lastChecked: Date.now()
    };
    await this.set(CONFIG.STORAGE_KEYS.CREDITS, creditsData);
    return creditsData;
  }

  // Pending Matches Queue
  async getPendingMatches() {
    const matches = await this.get(CONFIG.STORAGE_KEYS.PENDING_MATCHES);
    return matches || [];
  }

  async addPendingMatch(matchData) {
    const matches = await this.getPendingMatches();

    // Ensure transaction_id exists
    if (!matchData.transaction_id) {
      matchData.transaction_id = `tx_${Date.now()}_${Math.floor(Math.random() * 10000)}`;
    }

    matches.push({
      ...matchData,
      queuedAt: Date.now(),
      retryCount: 0
    });

    // Enforce absolute max to protect storage
    if (matches.length > MAX_PENDING_MATCHES) {
      matches.shift();
      this.logger.warn('Pending matches exceeded MAX_PENDING_MATCHES; dropping oldest item.');
    }

    await this.set(CONFIG.STORAGE_KEYS.PENDING_MATCHES, matches);
    this.logger.info(`Match queued. Queue size: ${matches.length}`);

    // Notify background if operational threshold exceeded
    if (matches.length > QUEUE_STOP_THRESHOLD) {
      try {
        if (typeof chrome !== 'undefined' && chrome.runtime) {
          chrome.runtime.sendMessage({ type: 'AUTOMATCH_QUEUE_EXCEEDED', length: matches.length }).catch(() => {});
        }
        this.logger.warn('Queue exceeded stop threshold — notified background to pause system.');
      } catch (e) {
        this.logger.warn('Failed to notify background about queue threshold', e);
      }
    }

    return matches;
  }

  async removePendingMatch(transactionId) {
    const matches = await this.getPendingMatches();
    const filtered = matches.filter(m => m.transaction_id !== transactionId);
    await this.set(CONFIG.STORAGE_KEYS.PENDING_MATCHES, filtered);
    return filtered;
  }

  async updatePendingMatch(transactionId, updates) {
    const matches = await this.getPendingMatches();
    const index = matches.findIndex(m => m.transaction_id === transactionId);
    if (index !== -1) {
      matches[index] = { ...matches[index], ...updates };
      await this.set(CONFIG.STORAGE_KEYS.PENDING_MATCHES, matches);
    }
    return matches;
  }

  // Settings Management
  async getSettings() {
    const settings = await this.get(CONFIG.STORAGE_KEYS.SETTINGS);
    return settings || {
      autoInject: CONFIG.FEATURES.AUTO_INJECT,
      debugMode: CONFIG.FEATURES.DEBUG_MODE,
      verboseLogging: CONFIG.FEATURES.VERBOSE_LOGGING
    };
  }

  async setSetting(key, value) {
    const settings = await this.getSettings();
    settings[key] = value;
    await this.set(CONFIG.STORAGE_KEYS.SETTINGS, settings);
    return settings;
  }

  // Sync Status
  async getLastSync() {
    return await this.get(CONFIG.STORAGE_KEYS.LAST_SYNC) || {};
  }

  async updateLastSync(type) {
    const lastSync = await this.getLastSync();
    lastSync[type] = Date.now();
    await this.set(CONFIG.STORAGE_KEYS.LAST_SYNC, lastSync);
    return lastSync;
  }
}

export default new StorageManager();