/**
 * AutoMatch Pro - Logger Utility
 * Provides structured logging with different levels
 * @module logger
 */

import { CONFIG } from './constants.js';

class Logger {
  constructor(context = 'AutoMatch') {
    this.context = context;
    this.levels = {
      DEBUG: 0,
      INFO: 1,
      WARN: 2,
      ERROR: 3
    };
    this.currentLevel = CONFIG.FEATURES.VERBOSE_LOGGING ? 0 : 1;
  }

  formatMessage(level, message, data = null) {
    const timestamp = new Date().toISOString();
    const prefix = `[${timestamp}] [${level}] [${this.context}]`;
    
    if (data) {
      return { prefix, message, data };
    }
    return `${prefix} ${message}`;
  }

  debug(message, data = null) {
    if (this.currentLevel <= this.levels.DEBUG) {
      const formatted = this.formatMessage('DEBUG', message, data);
      if (data) {
        console.log(formatted.prefix, formatted.message, formatted.data);
      } else {
        console.log(formatted);
      }
    }
  }

  info(message, data = null) {
    if (this.currentLevel <= this.levels.INFO) {
      const formatted = this.formatMessage('INFO', message, data);
      if (data) {
        console.info(formatted.prefix, formatted.message, formatted.data);
      } else {
        console.info(formatted);
      }
    }
  }

  warn(message, data = null) {
    if (this.currentLevel <= this.levels.WARN) {
      const formatted = this.formatMessage('WARN', message, data);
      if (data) {
        console.warn(formatted.prefix, formatted.message, formatted.data);
      } else {
        console.warn(formatted);
      }
    }
  }

  error(message, error = null) {
    const formatted = this.formatMessage('ERROR', message, error);
    if (error) {
      console.error(formatted.prefix, formatted.message, error);
    } else {
      console.error(formatted);
    }
    
    // Send error to background script for tracking
    try {
      chrome.runtime.sendMessage({
        type: 'ERROR_LOG',
        context: this.context,
        message,
        error: error ? error.toString() : null
      });
    } catch (e) {
      // Ignore if we can't send to background
    }
  }

  setLevel(level) {
    if (this.levels[level] !== undefined) {
      this.currentLevel = this.levels[level];
    }
  }
}

export default Logger;